from io import StringIO
from configparser import ConfigParser
import os
from pathlib import Path
from typing import Iterable
from .dict_utils import get_nested_key
from .settings import CLARION_BASE_DIRECTORY_DIR, CLARION_DIRECTORY_DIR
from .typehints import DirectoryKeyMap, DerivedField

SYDENT_CFG_FILE = f'{CLARION_DIRECTORY_DIR}/sydent.conf'


# ----------------------------------------------------------------------------
class BaseDirectoryConfig(ConfigParser):
    config_settings = SYDENT_CFG_FILE

    DERIVED_FIELDS: Iterable[DerivedField] = ()
    HOMESERVER_DIRECTORY_KEY_MAP: Iterable[DirectoryKeyMap] = ()

    def __init__(self, *args, **kwargs):
        self.config_settings = kwargs.pop('config_settings', '') or self.config_settings

        super().__init__(*args, **kwargs)
        self._values = {}
        self._loaded = False

    def load(self):
        """
        Load the ConfigParser object by calling the correct "read" method
        based on the value of `config_settings`.

        `config_settings` can be a string file path, dict, or StringIO
        """
        if self._loaded:
            return self

        if isinstance(self.config_settings, str):
            if Path(self.config_settings).exists():
                self.read(self.config_settings)
            else:
                self.read(f'{CLARION_BASE_DIRECTORY_DIR}/sydent.conf')

        # Enable easy testing by using StringIO object
        if isinstance(self.config_settings, StringIO):
            self.read_file(self.config_settings)

        if isinstance(self.config_settings, dict):
            self.read_dict(self.config_settings)

        self._loaded = True

        return self

    def set_value(self, key, value) -> None:
        if not self.has_section(self.main_section):
            self.add_section(self.main_section)
        self.set(self.main_section, key, value=str(value))

    def get_value(self, key):
        if not self.has_section(self.main_section):
            self.add_section(self.main_section)
        return self.get(self.main_section, key)

    def save(self) -> None:
        """
        Save entire config file to disk
        """
        os.makedirs(CLARION_DIRECTORY_DIR, exist_ok=True)
        with open(SYDENT_CFG_FILE, 'w') as cfg:
            self.write(cfg)

    def update_values(self, data):
        """
        Update config settings with supplied data dict, which
         usually comes from defining the Homeserver settings.
        """
        cfg = self.load()

        for field in self.DERIVED_FIELDS:
            cfg.set_value(field.name, field.value.format(**data))

        for field in cfg.HOMESERVER_DIRECTORY_KEY_MAP:
            value = str(get_nested_key(data, field.homeserver, default=''))
            cfg.set_value(field.directory, value)

        self.save()


class DirectoryDefaults(BaseDirectoryConfig):
    main_section = 'DEFAULT'


class DirectoryGeneral(BaseDirectoryConfig):
    main_section = 'general'

    DERIVED_FIELDS = [
        DerivedField('server.name', '{subdomain}.clarion.directory', True),
    ]


class DirectoryEmail(BaseDirectoryConfig):
    main_section = 'email'

    DERIVED_FIELDS = [
        DerivedField('email.default_web_client_location', 'https://{subdomain}-web.{clarion_service}/e/', True),
    ]

    HOMESERVER_DIRECTORY_KEY_MAP = (
        DirectoryKeyMap('email:smtp_host', 'email.smtphost'),
        DirectoryKeyMap('email:smtp_host', 'email.hostname'),
        DirectoryKeyMap('email:smtp_port', 'email.smtpport'),
        DirectoryKeyMap('email:smtp_pass', 'email.smtppassword'),
        DirectoryKeyMap('email:smtp_user', 'email.smtpusername'),
        DirectoryKeyMap('email:notif_from', 'email.from'),
    )
