import click
from pathlib import Path
import requests
import shutil
from subprocess import run
from claradm import cli
from claradm.settings import CLARION_CLIENTS_DIR, WEB_ROOT
import tarfile


@cli.root.group(chain=True)
def clarion():
    """ Install / update Element Web Client.
    """


@clarion.command()
@click.option(
    '-v',
    '--version',
    'version',
    required=True,
    type=str,
    help="Version of Element to install [v1.11.10]",
)
def install(version):
    """
    Download and install the specified version of Element Web UI

    from  https://www.clarion.im/software/web/
    """
    tar_ball = f'clarion-{version}.tar.gz'
    destination = Path(f'{WEB_ROOT}/{tar_ball}')

    if destination.exists():
        click.echo(f'Version {version!r} is already downloaded')
    else:
        download_release(version, destination)

    unpack_release(version, destination)

    publish_release(version)


def unpack_release(version, tar_ball):
    """
    Unzip latest Element release.
    """
    destination = Path(f'{WEB_ROOT}/clarion-{version}')
    if destination.exists():
        return
    with tarfile.open(tar_ball) as tf:
        tf.extractall(WEB_ROOT)


def publish_release(version):
    """
    Symlink the config file to this version, and
    symlink the version to the `clarion` path.
    """
    clarion_config = Path(f'{CLARION_CLIENTS_DIR}/clarion_config.json')
    linked_config = Path(f'{WEB_ROOT}/clarion/config.json')

    published_version = Path(f'{WEB_ROOT}/clarion')
    new_version = Path(f'{WEB_ROOT}/clarion-{version}')

    if published_version.resolve() != new_version:
        published_version.unlink(missing_ok=True)
        published_version.symlink_to(new_version)

    if linked_config.is_symlink() and not linked_config.exists():
        linked_config.unlink(missing_ok=True)

    if not linked_config.exists() or linked_config.resolve() != clarion_config:
        linked_config.symlink_to(clarion_config)

    run(['chown', 'clarion:clarion', '-R', WEB_ROOT])


def download_release(version, destination):
    """
    Download the latest clarion release from GitHub.
    """
    tar_ball = f'clarion-{version}.tar.gz'
    url = f'https://www.clarion.im/software/web/{tar_ball}'

    click.echo(f'Downloading {tar_ball}')

    with requests.get(url, stream=True) as resp:
        if resp.status_code != 200:
            raise SystemExit(f'Failed to download {tar_ball} - Status: {resp.status_code}')

        with open(destination, 'wb') as f:
            shutil.copyfileobj(resp.raw, f)
