# Helper commands to streamline the setup of a brand new Clarion Server.
from aspen_crypto.encode import mask, unmask
import click
import hashlib
import hmac
import json
from lcrequests.sockets import LCRequestUnixSocket
import secrets
from typing import Iterable
from uuid import UUID
import yaml

from claradm import cli
from claradm.settings import ADMIN_USERNAME


@cli.root.group(chain=True)
def account():
    """ Create new user accounts.
    """


@account.command(name='add')
@click.option(
    '-u',
    '--username',
    'username',
    required=True,
    type=str,
    help="Username prefix: <username>:clarion.im"
)
@click.option('-n', '--name', 'name', required=True, type=str, help="Person's Full Name")
@click.option('-a', '--admin', 'admin', default=False, type=bool, help="Person is server admin")
@click.option('-c', '--clavis', 'clavis', required=False, type=UUID, help="Clavis SSO Canonical ID")
@click.option(
    '-e',
    '--email',
    'email',
    required=False,
    type=str,
    default=[],
    multiple=True,
    help='Email addresses to add as Third Party Identifier. (Accepts multiple parameters)'
)
@click.option(
    '-m',
    '--mobile',
    'mobile',
    required=False,
    type=str,
    default=[],
    multiple=True,
    help='Mobile Phone Number to add as Third Party Identifier. (Accepts multiple parameters)'
)
def register(username, name, admin, clavis, email, mobile):
    """
    Register new user on Clarion Messaging Server.

    -u thom -n 'Thom Smith' -a False -c 65bf430b-09d1-11ed-b1e5-7085c27cfb9b -e thom@smith.net -e thom@smith.com -m 16188351234
    """
    password = click.prompt('Secure Password', hide_input=True)
    _register_user(username, name, password, admin, clavis, email, mobile)


@account.command()
def admin():
    """
    Register default admin user on Clarion Messaging Server.
    """
    try:
        server_name = homeserver_name()
        user = f'@{ADMIN_USERNAME}:{server_name}'
        token = get_access_token(is_admin=True, data=None, user=user)
    except (FileNotFoundError, KeyError, SystemExit):
        token = None
        click.echo('Creating default admin user')

    # thinkwell.clarion.im
    config = load_config('homeserver.yaml')

    if token:
        click.echo('Default admin user already exists')
        update_claradm_config(token, config)
        return

    email_username = config['subdomain']
    hostname = config['clarion_service']
    name = 'Clarion Admin'
    password = secrets.token_urlsafe()

    success = _register_user(
        ADMIN_USERNAME,
        name,
        password,
        admin=True,
        clavis='',
        email=[f'{email_username}@{hostname}'],
        mobile=(),
    )
    if success:
        persist_admin_password(password)

    update_claradm_config(token, config)


def update_claradm_config(token, config):
    """
    Create or update claradm client config from server config settings.
    """

    from . import APIHelper
    helper = APIHelper(
        config_path='~/.config/claradm.yaml',
        verbose=1,
        # batch=click.echo,
        no_confirm=False,
        output_format_cli='yaml',
    )
    user = f'{ADMIN_USERNAME}:{config["server_name"]}'
    helper.write_config({
        'homeserver': config['server_name'],
        'user': user,
        'token': token or get_access_token(is_admin=True, data=None, user=user),
        'base_url': config['server_name'] + ".clarion.im",
        'admin_path': '/_synapse/admin',
        'matrix_path': '/_matrix',
        'format': 'yaml',
        'timeout': 30,
        'server_discovery': 'well-known',
    })


def generate_mac(nonce, user, password, admin=False, user_type=None, shared_secret=''):
    """
    Generate HMAC token for use with Shared-Secret Registration
    """
    key = shared_secret or get_conf_param('registration_shared_secret', 'homeserver.yaml')
    mac = hmac.new(key=key.encode(), digestmod=hashlib.sha1)

    mac.update(nonce.encode("utf8"))
    mac.update(b"\x00")
    mac.update(user.encode("utf8"))
    mac.update(b"\x00")
    mac.update(password.encode("utf8"))
    mac.update(b"\x00")
    mac.update(b"admin" if admin else b"notadmin")
    if user_type:
        mac.update(b"\x00")
        mac.update(user_type.encode("utf8"))

    return mac.hexdigest()


def load_config(config_file):
    """
    Load config variable from specified file
    """
    with open(f'/etc/clarion/homeserver/conf.d/{config_file}') as cf:
        return yaml.safe_load(cf)


def get_conf_param(key, config_file):
    """
    Load config variable from specified file
    """
    return load_config(config_file)[key]


def homeserver_name():
    return get_conf_param('server_name', 'homeserver.yaml')


def get_admin_password():
    """
    Load admin password from config file.
    """
    pwd = get_conf_param('admin_password', 'credentials.yaml')
    return unmask(pwd)


def persist_admin_password(password):
    """
    Save admin password to config file.
    """
    cfg_file = f'/etc/clarion/homeserver/conf.d/credentials.yaml'
    try:
        with open(cfg_file) as cf:
            cfg = yaml.safe_load(cf)
    except FileNotFoundError:
        cfg = {}

    cfg['admin_password'] = mask(password)

    with open(cfg_file, 'w') as cf:
        yaml.safe_dump(cfg, cf)


def get_nonce():
    """
    Fetch one-time nonce from synapse server,
    for including in body to register new user.
    """
    conn = LCRequestUnixSocket(unix_socket='/var/run/clarion/homeserver.sock')
    conn.request("GET", "/_synapse/admin/v1/register")
    resp = conn.getresponse()
    resp_data = resp.json()

    try:
        return resp_data['nonce']
    except KeyError:
        pass

    if resp.status >= 500:
        raise SystemExit('Unexpected Server Error Occurred')

    try:
        raise SystemExit(f"\n{resp_data['error']}\n")
    except Exception:
        raise SystemExit(f'\nAccess Token lookup failed: {resp_data}\n')


def add_3pids_sso(
    user_id: str,
    admin: bool,
    access_token: str,
    sso_id: str = '',
    emails=(),
    mobiles=(),
):
    """
    Add SSO external IDs and third party identifiers.
    """
    data = {'admin': admin}
    threepids = []
    for email in emails:
        threepids.append({'address': email, 'medium': 'email'})

    for mobile in mobiles:
        threepids.append({'address': mobile, 'medium': 'msisdn'})

    if threepids:
        data['threepids'] = threepids
    if sso_id:
        data['external_ids'] = [
            {
                'external_id': sso_id,
                'auth_provider': 'oidc-clavis'
            },
        ]

    if not data:
        click.echo('No data provided for SSO IDs or 3PIDs')
        return

    headers = {'Authorization': f'Bearer {access_token}'}
    url = f'/_synapse/admin/v2/users/{user_id}'
    data = json.dumps(data, default=str)

    conn = LCRequestUnixSocket(unix_socket='/var/run/clarion/homeserver.sock')
    conn.request("PUT", url=url, headers=headers, body=data)
    resp = conn.getresponse()

    if resp.status >= 300:
        click.echo(f'SSO / 3PIDs failed - {resp.status} - {json.loads(resp.read())}')


def get_access_token(is_admin, data: None, user, password=''):
    """
    Get an access token for the main admin user.
    """
    if is_admin:
        data = data or {}
        try:
            return data['access_token']
        except KeyError:
            pass

    data = {
        'type': 'm.login.password',
        'user': user,
        'password': password or get_admin_password(),
    }

    conn = LCRequestUnixSocket(unix_socket='/var/run/clarion/homeserver.sock')
    conn.request("POST", "/_matrix/client/v3/login", json.dumps(data))
    resp = conn.getresponse()
    resp_data = resp.json()

    try:
        return resp_data['access_token']
    except KeyError:
        pass

    if resp.status >= 500:
        raise SystemExit('Unexpected Server Error Occurred')

    try:
        raise SystemExit(f"\nget_access_token: {resp_data['error']}\n")
    except Exception:
        raise SystemExit(f'\nAccess Token lookup failed: {resp_data["content"]}\n')


def _register_user(
    username: str,
    name: str,
    password: str,
    admin: bool,
    clavis: str,
    email: Iterable[str],
    mobile: Iterable[str],
):
    """
    Register new user on Synapse
    """
    url = f'/_synapse/admin/v1/register'

    nonce = get_nonce()
    mac_digest = generate_mac(
        nonce=nonce,
        user=username,
        password=password,
        admin=admin,
        user_type=None,
    )

    data = {
        "nonce": nonce,
        "username": username,
        "displayname": name,
        "password": password,
        "mac": mac_digest,
        "admin": admin,
        "user_type": None,
    }

    conn = LCRequestUnixSocket(unix_socket='/var/run/clarion/homeserver.sock')
    conn.request("POST", url=url, body=json.dumps(data))
    resp = conn.getresponse()
    server = 'shalom.clarion.im'

    resp_data = resp.json()

    if resp.status >= 500:
        raise SystemExit('Unexpected Server Error Occurred')

    if resp.status != 200:
        try:
            click.echo(f"\n{resp_data['error']}\n")
        except Exception:
            click.echo(f'Registration failed - {resp_data}')
        return False

    if any([clavis, email, mobile]):
        user_id = resp_data['user_id']
        user = f'@{username}:{server}'
        token = get_access_token(admin, resp_data, user, password=password)
        add_3pids_sso(user_id, admin, token, clavis, email, mobile)

    click.echo(f'\nAdded @{username} to {server}\n')

    return True
