from pathlib import Path
import psycopg
from invoke import Context, task

from ..backups import (
    BackupDatabase,
    ObjectStorageFile,
    object_server_client,
    latest_database_backups,
)
from .services import start, stop
from ..settings import DATABASE_LAST_BACKUP_TIME, S3_REGION
from ..typehints import S3_REGIONS
from ..utils.databases import get_databases, prepare_console_database, restore_database
from ..utils.zabbix import record_successful_backup_time


@task(
    help={
        'interval': 'File archive interval - (hourly, daily, weekly, monthly)',
        'region': f'Region for saving the archive file. Default: {S3_REGION!r}',
    },
)
def backup(ctx: Context, interval='', region: S3_REGIONS = S3_REGION) -> bool:
    """
    Backup all databases in the local Postgres cluster.
    """
    client = object_server_client(region)
    for database in get_databases():
        print(f'{interval.title()} backup of database: {database!r}.')
        BackupDatabase(ctx, filename=f'{database}.db', interval=interval, client=client).process()
    record_successful_backup_time(DATABASE_LAST_BACKUP_TIME)
    return True


@task(
    help={
        'name': 'Database name',
        'interval': 'File archive interval - (hourly, daily, weekly, monthly)',
        'region': f'Region for saving the archive file. Default: {S3_REGION!r}',
    },
)
def backup_files(ctx: Context, name='', interval='', region: S3_REGIONS = S3_REGION) -> bool:
    """
    Show all database backup files for specified database name and interval.
    """
    client = object_server_client(region)
    backups = BackupDatabase(ctx, filename='', interval='', client=client).backups(name, interval)

    for interval, db_files in backups.items():
        print(f'\n{interval.title()} backup files')
        for bf in db_files:
            print(bf)
        print()
    return True


@task(
    name='list',
    help={
        'interval': 'File archive interval - (hourly, daily, weekly, monthly)',
        'region': f'Region for saving the archive file. Default: {S3_REGION!r}',
    },
)
def db_list(ctx: Context, interval, region: S3_REGIONS = S3_REGION) -> None:
    """
    Show all databases in the cluster that have backups for specified interval.
    """
    client = object_server_client(region)
    try:
        databases = BackupDatabase(
            ctx,
            filename='',
            interval=interval,
            client=client,
        ).db_list(interval)
    except Exception as e:
        raise SystemExit(f'Unable to get database list - Error:\n\n{e}\n')

    if not databases:
        print(f'\nNo {interval.title()} database backups found\n')
        return

    print(f'\nDatabases with {interval.title()} backups')
    for db in databases:
        print(db)
    print()


@task(
    pre=[stop],
    post=[start],
    help={
        'name': 'Database name to restore',
        'filename': 'Backup filename to download',
        'interval': 'File archive interval - (hourly, daily, weekly, monthly)',
        'region': f'Region for saving the archive file. Default: {S3_REGION!r}',
    },
)
def restore(ctx: Context, name, filename, interval, region: S3_REGIONS = S3_REGION) -> None:
    """
    Dump database (if existing) and create from backup file downloaded from the backup server.
    """
    client = object_server_client(region)

    try:
        dst = ObjectStorageFile(filename, prefix=f'databases/{interval}', client=client).download()
    except Exception as e:
        raise SystemExit(f'Unable to download {name!r} - Error:\n\n{e}\n')

    if not Path(dst).exists():
        print(f'Unable to download {filename} and restore database {name!r} from backup.')
        return

    restore_database(ctx, name, dst)


@task(
    pre=[stop],
    post=[start],
    help={
        'region': f'Region for saving the archive file. Default: {S3_REGION!r}',
    },
)
def restore_all(ctx: Context, region: S3_REGIONS = S3_REGION) -> None:
    """
    Download & restore the latest backup of all databases for this system.
    """
    # Create the "logcabin" role in case this is restoring legacy databases.
    with psycopg.connect('user=postgres') as conn:
        with conn.cursor() as cursor:
            try:
                cursor.execute('CREATE ROLE logcabin WITH SUPERUSER LOGIN;')
            except psycopg.errors.DuplicateObject:
                pass

    client = object_server_client(region)

    backups_retrieved = []
    for database in latest_database_backups(client):
        downloaded_backup_file = ObjectStorageFile(
            database.file_path,
            prefix=BackupDatabase.base_prefix,
            client=client,
        ).download()
        if Path(downloaded_backup_file).exists():
            backups_retrieved.append((database.name, downloaded_backup_file))

    for db_name, backup_file in backups_retrieved:
        restore_database(ctx, db_name, backup_file)

    prepare_console_database(ctx)


__all__ = (
    'backup',
    'backup_files',
    'db_list',
    'restore',
    'restore_all',
)
