import arrow
from collections import defaultdict
from minio import Minio
import os
from pathlib import Path
import pwd

from .object_storage_server import ObjectStorageFile
from capsule import settings


# --------------------------------------------------------------------------------
class BackupFile:
    """
    Dump a database or compress a directory to file and backup to remote server.

    Create daily / weekly / monthly backup files, and
    maintain designated archive of copies on the remote server
    """

    base_prefix = ''
    file_path = ''
    exclude_patterns: tuple[str, ...] = ()

    # Scrub old files after after file limit exceeded
    scrub_local_files = True
    scrub_remote_files = True

    def __init__(self, ctx, filename: str, interval: str = '', client: Minio | None = None) -> None:
        self.ctx = ctx
        self.base_filename, self.extension = os.path.splitext(filename)
        self.date = arrow.now()
        self._filename = ''
        self._interval = interval
        self.server = ObjectStorageFile(
            prefix=f'{self.base_prefix}/{self.interval()}',
            filename=self.filename(),
            client=client,
        )

    def exists(self) -> bool:
        """
        Check if the file name exists on the file system.
        """
        return os.path.exists(self.filename())

    def process(self) -> None:
        """
        Perform all file preparations and upload to Object Storage server.
        Override on subclasses to add additional processing.
        """
        self.upload()
        self.scrub()
        self.chown()

    def interval(self) -> str:
        """
        Calculate whether this is a daily / weekly / monthly backup file.
        """
        if not self._interval:
            tomorrow = self.date.shift(days=1)
            saturday_weekday = 5

            if self.date.month != tomorrow.month:
                self._interval = 'monthly'
            elif self.date.weekday() == saturday_weekday:
                self._interval = 'weekly'
            else:
                self._interval = 'daily'

        return self._interval

    def max_age(self) -> tuple[int, arrow.Arrow]:
        """
        Return oldest date of any backup files that should be retained.
        All files older than this date should be deleted.
        """
        interval = self.interval()
        HOURLY_MAX_FILE_COUNT = settings.HOURLY_MAX_FILE_COUNT
        DAILY_MAX_FILE_COUNT = settings.DAILY_MAX_FILE_COUNT
        MONTHLY_MAX_FILE_COUNT = settings.MONTHLY_MAX_FILE_COUNT
        WEEKLY_MAX_FILE_COUNT = settings.WEEKLY_MAX_FILE_COUNT

        if interval == 'monthly':
            return MONTHLY_MAX_FILE_COUNT, self.date.shift(months=-MONTHLY_MAX_FILE_COUNT)

        if interval == 'weekly':
            return WEEKLY_MAX_FILE_COUNT, self.date.shift(weeks=-WEEKLY_MAX_FILE_COUNT)

        if interval == 'hourly':
            return HOURLY_MAX_FILE_COUNT, self.date.shift(hours=-HOURLY_MAX_FILE_COUNT)

        return DAILY_MAX_FILE_COUNT, self.date.shift(days=-DAILY_MAX_FILE_COUNT)

    def filename(self) -> str:
        """
        Calculate backup file name from date, interval type and base file name.
        """
        if not self._filename:
            if self.interval() == 'hourly':
                backup_date = self.date.strftime('%Y%m%d-%H')
            else:
                backup_date = self.date.strftime('%Y%m%d')
            self._filename = f'{self.file_path}/{self.base_filename}-{backup_date}{self.extension}'
        return self._filename

    def chown(self) -> None:
        """
        Set the file owner and group to maintain consistency
        when backups are created as "root" user.
        """
        try:
            pwd.getpwnam(settings.FILE_OWNER)
            self.ctx.run(
                f'chown -R {settings.FILE_OWNER}:{settings.FILE_GROUP} {self.file_path}',
                pty=True,
            )
        except KeyError:
            pass

    def upload(self) -> None:
        """
        Upload current file to remote server.
        """
        self.server.upload()

    def scrub(self) -> None:
        """
        Scrub remote / local backup files after age limit is exceeded.
        """
        max_file_count, max_age = self.max_age()
        obsolete_ts = int(max_age.timestamp())

        if self.scrub_local_files:
            pattern = f'{self.base_filename}-*{self.extension}'
            backup_files = list(Path(self.file_path).glob(pattern))

            # Archive interval not considered for local backup files.
            if len(backup_files) > settings.DAILY_MAX_FILE_COUNT:
                for backup_file in backup_files:
                    if backup_file.stat().st_mtime < obsolete_ts:
                        backup_file.unlink(missing_ok=True)

        if self.scrub_remote_files:
            self.server.scrub(max_file_count, max_age)

    def backups(self, name: str = '', interval: str = '') -> dict[str, list[str]]:
        """
        Get a dictionary of all backup files in self.base_prefix, grouped by interval.
        """
        all_backups = defaultdict(list)
        prefix = f'{self.base_prefix}/{interval}' if interval else self.base_prefix

        for backup in self.server.list_objects(prefix=prefix):
            backup_file = backup.object_name.split('/')
            backup_interval = backup_file[1]
            db_name = backup_file[-1].split('-')[0]
            if not name or db_name == name:
                all_backups[backup_interval].append(backup_file[-1])

        return all_backups


__all__ = [
    'BackupFile',
]
