#!/usr/bin/python3

import io
import os
from os.path import dirname, join
import re
from setuptools import setup


def read(*names, **kwargs):
    path = join(dirname(__file__), *names)
    return io.open(path, encoding=kwargs.get('encoding', 'utf8')).read()


def get_packages(package):
    """
    Return root package and all sub-packages.
    """
    return [
        dirpath
        for dirpath, dirnames, filenames in os.walk(package)
        if os.path.exists(os.path.join(dirpath, '__init__.py'))
    ]


def get_package_data(package):
    """
    Return all files under the root package, that are not in a
    package themselves.
    """
    walk = [
        (dirpath.replace(package + os.sep, '', 1), filenames)
        for dirpath, dirnames, filenames in os.walk(package)
        if not os.path.exists(os.path.join(dirpath, '__init__.py'))
    ]

    filepaths = []
    for base, filenames in walk:
        filepaths.extend([os.path.join(base, filename) for filename in filenames])
    return {package: filepaths}


def get_version():
    with open('capsule/__init__.py', 'r') as f:
        line = f.readline()
        match = re.match(r'__version__ = \'([\d\.]+)\'', line)

        if not match:
            raise ImportError("Can't read the version of Capsule")

        version = match.group(1)
        return version


setup(
    name='capsule',
    version=get_version(),
    license='SAP',
    description='Capsule Backup Center',
    author='Dave Burkholder',
    author_email='dave@compassfoundation.io',
    url='https://code.compassfoundation.io/dave/capsule',
    packages=get_packages('capsule'),
    zip_safe=True,
    options={
        'bdist_rpm': {
            'post_install': 'post_install.sh',
            'pre_install': 'pre_install.sh',
        },
    },
    classifiers=[
        'Intended Audience :: Developers',
        'License :: SAP',
        'Operating System :: Unix',
        'Operating System :: POSIX',
        'Programming Language :: Python :: 3.11',
        'Programming Language :: Python :: 3.12',
        'Programming Language :: Python :: Implementation :: CPython',
        'Programming Language :: Python :: Implementation :: PyPy',
        'Topic :: Utilities',
    ],
    keywords=[
        'DrawBridge Console',
        'Backup center',
    ],
    install_requires=[
        'arrow',
        'console_keyring',
        'invoke',
        'lcconfig',
        'minio',
        'psycopg',
        'system_env',
        'py-zabbix',
    ],
    entry_points={
        'console_scripts': [
            'capsule = capsule.main:program.run',
        ],
    },
)
