from datetime import datetime, timezone
import logging
from pyzabbix import ZabbixMetric, ZabbixSender
import socket
from pathlib import Path
from typing import Iterable

logger = logging.getLogger(__name__)


def zabbix(packets: Iterable[tuple]):
    """
    Send metrics back to Zabbix server.
    """
    dbos_location = '/run/zabbix/zabbix-agent2.conf'
    if Path(dbos_location).exists():
        use_config = dbos_location
    elif Path('/etc/zabbix/zabbix_agent2.conf').exists():
        use_config = '/etc/zabbix/zabbix_agent2.conf'
    else:
        use_config = '/etc/zabbix/zabbix_agentd.conf'

    zabbix_key = get_hostname_key(use_config)

    packet = [ZabbixMetric(zabbix_key, key, str(value)) for key, value in packets]

    try:
        ZabbixSender(use_config=use_config).send(packet)
    except FileNotFoundError:
        logger.error(
            'Backup tracking not possible - no Zabbix configuration file found! '
            'Consider installing and configuring Zabbix.'
        )
    except Exception as e:
        logger.exception(f'Unable to send Zabbix packets: {e}')
        print(f'Unable to send Zabbix packets: {e}')


def get_hostname_key(zabbix_config: str):
    """
    Read the local hostname from the Zabbix config file.
    Hostname=thinkwell.drawbridge.systems
    """
    try:
        with open(zabbix_config) as zc:
            for line in zc:
                if line.startswith('Hostname'):
                    return line.partition('=')[-1].strip()
    except FileNotFoundError:
        pass
    except Exception as e:
        logger.exception('Unable to read Zabbix config file: %s', e)
    return socket.gethostname()


def record_successful_backup_time(zabbix_item_key: str):
    """
    Post the latest successful backup time
    to the relevant Zabbix item.
    """
    now = datetime.now(timezone.utc)
    zabbix([(zabbix_item_key, int(now.timestamp()))])


__all__ = [
    'zabbix',
    'record_successful_backup_time',
]
