from invoke import Context
from invoke.exceptions import ThreadException, Failure
from typing import Sequence

import psycopg
from psycopg import connect as db_connect
from ..settings import APPS_ROOT, DATABASE_NAME, DB_HOST, IS_DRAWBRIDGE_OS
from ..typehints import DatabaseStatus


# --------------------------------------------------------------------------------
def get_databases(excluded: Sequence[str] = ()) -> list[str]:
    """
    Get list of databases created on the system.

    ['logcabin', 'clarion', 'odoo']
    """
    excluded_dbs = {'excluded': excluded or ('template0', 'template1', 'postgres')}
    sql = """
    SELECT datname FROM pg_database 
    WHERE datname NOT IN %(excluded)s;
    """
    with psycopg.connect(f"user=postgres host={DB_HOST}") as conn:
        with conn.cursor() as cursor:
            cursor.execute(sql % excluded_dbs)

            databases = []
            for db in cursor.fetchall():
                name = db[0]
                if isinstance(name, bytes):
                    databases.append(name.decode())
                else:
                    databases.append(name)

            return databases


# --------------------------------------------------------------------------------
def create_console_database():
    """
    Wrapper function to for portal database creation.
    """
    return create_database('drawbridge', check_existing=True)


# --------------------------------------------------------------------------------
def create_database(name: str, check_existing: bool = True):
    """
    Ensure that the console database is created.
    """
    if check_existing:
        if name in get_databases():
            return DatabaseStatus(exists=True, created=False)

    try:
        conn = db_connect(f"user=postgres port=5432 host={DB_HOST}")
        conn.autocommit = True
        cursor = conn.cursor()
        cursor.execute("CREATE DATABASE %s;" % name)
        return DatabaseStatus(exists=True, created=True)
    except Exception as e:
        raise SystemExit(f'Unable to create database. {e}.')


# --------------------------------------------------------------------------------
def drop_database(database_name: str):
    """
    Drop the database with the specified name.
    """
    try:
        conn = db_connect(f"user=postgres port=5432 host={DB_HOST}")
        conn.autocommit = True
        cursor = conn.cursor()
        cursor.execute('DROP DATABASE IF EXISTS %s;' % database_name)
    except Exception as e:
        raise SystemExit(f'Unable to drop {database_name}. {e}.')

    if conn.closed == 0:
        conn.close()


# --------------------------------------------------------------------------------
def prepare_console_database(ctx: Context):
    """
    Ensure that Console database triggers are configured
    and database owner is set correctly
    """
    ctx.cd(APPS_ROOT)
    for cmd in (
        'python3 manage.py pgtrigger install',
        f'psql -U postgres -d {DATABASE_NAME} -f apps/accounts/sql/ltree_triggers_policy.sql',
        f'psql -U postgres -d {DATABASE_NAME} -f apps/categories/sql/ltree_triggers_category.sql',
    ):
        try:
            ctx.run(cmd, hide='both')
        except (OSError, Failure, ThreadException) as e:
            print(f'Error running cmd: {cmd}.\n{e}')

    if not IS_DRAWBRIDGE_OS:
        return

    with psycopg.connect(f'dbname={DATABASE_NAME} user=postgres') as conn:
        with conn.cursor() as cursor:
            cursor.execute('REASSIGN OWNED BY logcabin TO %(db_name)s' % {'db_name': DATABASE_NAME})

    with psycopg.connect(f'dbname={DATABASE_NAME} user=postgres') as conn:
        with conn.cursor() as cursor:
            cursor.execute('DROP ROLE IF EXISTS logcabin;')


# --------------------------------------------------------------------------------
def restore_database(ctx: Context, name: str, backup_file: str):
    """
    Restore database from backup file.
    """
    if name in get_databases():
        print(f'\nDrop existing {name!r} database')
        drop_database(name)

    create_database(name, check_existing=False)

    print(f'Restoring {name!r} database from backup {backup_file!r}\n')
    try:
        ctx.run(f'pg_restore -U postgres -d {name} {backup_file}', hide='both')
    except (OSError, Failure, ThreadException):
        # pg_restore commands can print out non-zero errors,
        # but don't consider that a restore failure.
        pass


__all__ = (
    'get_databases',
    'create_database',
    'drop_database',
    'prepare_console_database',
    'restore_database',
)
