from invoke import Context, task

from ..backups import BackupEtc
from ..downloader import downloader
from ..settings import DIRECTORY_LAST_BACKUP_TIME
from ..typehints import INTERVALS
from ..utils.zabbix import record_successful_backup_time


@task
def backup(ctx: Context) -> bool:
    """
    Backup needed files from the /etc directory.
    """
    print('Backing up subsdirectories in /etc... ', end='', flush=True)
    BackupEtc(ctx, filename='etc.tbz2').process()
    print('Done')

    record_successful_backup_time(DIRECTORY_LAST_BACKUP_TIME)

    return True


@task(
    help={
        'interval': 'File archive interval - (daily, weekly, monthly)',
    },
)
def backup_files(ctx: Context, interval: INTERVALS = '') -> bool:
    """
    Show all Etc backup files for specified interval.
    """
    backups = BackupEtc(ctx, filename='', interval='').backups(interval=interval)

    for interval, db_backups in backups.items():  # type: ignore[assignment]
        print(f'\n{interval.title()} backup files')
        for bf in db_backups:
            print(bf)
        print()

    return True


@task(
    help={
        'interval': 'File archive interval - (daily, weekly, monthly)',
        'filename': 'Remote backup filename to download',
    },
)
def download(ctx: Context, interval: INTERVALS, filename: str) -> bool:
    """
    Download specified Etc backup tarball from the backup server.
    """
    downloader(interval, filename, BackupEtc.base_prefix)
    return True


__all__ = (
    'backup',
    'backup_files',
    'download',
)
