from lcconfig.configparser import ConsoleSettingsConfig
from getpass import getpass
from invoke import task, Context
from pathlib import Path
from capsule import settings
from capsule.typehints import KeyringService
from capsule.wizards import persist_object_storage_config
from console_keyring.console_keyring import ConsoleKeyring


@task
def setup(ctx: Context) -> None:
    """
    Setup object storage, system, and email config files for Compass systems.
    """
    if settings.IS_DRAWBRIDGE_OS:
        print('Is a Drawbridge system. Not overwriting existing System config and Email config.')
        return

    if settings.IS_GENERAL_SERVER:
        Path('/etc/compass').mkdir(exist_ok=True)

    ConsoleKeyring.generate_new_key()
    setup_object_storage_server()
    setup_boxdata()
    setup_email()


def setup_object_storage_server() -> None:
    """
    Setup Object Storage config file and add save secrets.
    """
    endpoint = (
        input('Enter Object Storage Server endpoint (garage.compassfoundation.net): ')
        or 'garage.compassfoundation.io'
    )

    region = input('Enter Object Server Region (ironwood): ') or 'ironwood'

    access_key = ''
    while not access_key:
        access_key = input('Enter access key: ')

    secret_key = ''
    while not secret_key:
        secret_key = getpass('Enter secret key: ')

    creds = {
        'access_key': access_key,
        'secret_key': secret_key,
    }
    persist_object_storage_config(
        endpoint,
        region,
        creds=creds,
        service=KeyringService.LocalConsole,
    )


def setup_boxdata() -> None:
    """
    Setup System data config file.
    """
    name = ''
    adminurl = ''

    while not name:
        name = input('Enter system name: ')
    while not adminurl:
        adminurl = input('Enter system admin URL: ')

    ConsoleSettingsConfig().save_section(
        section='BoxData',
        data={
            'name': name,
            'adminurl': adminurl,
        },
    )


def setup_email() -> None:
    """
    Setup email config file.
    """
    host = ''
    port = 0
    username = ''
    password = ''
    from_address = ''

    while not host:
        host = (
            input('Enter email server host (mail.drawbridge.systems): ')
            or 'mail.drawbridge.systems'
        )
    while not port or type(port) is not int:
        port_input = input('Enter email server port (587): ') or 587
        try:
            port = int(port_input)
        except ValueError:
            pass
    while not username:
        username = input('Enter username (name@drawbridge.systems): ')
    while not password:
        password = getpass('Enter password: ')
    while not from_address:
        from_address = input('Enter from address (name@drawbridge.systems): ')

    ConsoleSettingsConfig().save_section(
        section='default_email',
        data={
            'host': host,
            'port': port,
            'use_tls': True,
            'username': username,
            'password': password,
            'from_address': from_address,
        },
    )


__all__ = [
    'setup',
]
