import logging
import os

from .base import BackupFile
from ..settings import OPT_DIR, PORTAL_DIR
from ..utils.tar import create_tarball

logger = logging.getLogger(__name__)


# --------------------------------------------------------------------------------
class BackupDirectory(BackupFile):
    """
    Compress directory to bz2 tarball and backup to remote server.

    Create daily / weekly / monthly backup tarballs, and
    maintain designated archive of copies on the remote server
    """

    directories: tuple[str, ...] = ()

    def process(self) -> None:
        """
        Perform all directory tar, upload and cleanup operations.
        """
        self.tar()
        super().process()

    def tar(self) -> None:
        """
        Tar directory to local tarball
        """
        if self.exists():
            return

        if not os.path.exists(self.file_path):
            os.makedirs(self.file_path, exist_ok=True)

        create_tarball(self.directories, self.filename(), self.exclude_patterns)


class BackupCerts(BackupDirectory):
    bucket = 'cert_backups'
    file_path = f'{OPT_DIR}/{bucket}'
    base_prefix = 'certs'
    directories = ('/opt/aspen_portal/files',)


class BackupDBOSConfig(BackupDirectory):
    bucket = 'dbosbackups'
    file_path = f'{OPT_DIR}/{bucket}'
    base_prefix = 'dbos'
    directories = ('/config',)


class BackupEtc(BackupDirectory):
    bucket = 'etcbackups'
    file_path = f'{OPT_DIR}/{bucket}'
    base_prefix = 'etc'
    directories = (
        PORTAL_DIR,
        '/etc/centrifugo',
        '/etc/clarion',
        '/etc/letsencrypt',
        '/etc/nginx',
        '/etc/passageway',
    )
    exclude_patterns = (
        r'\.mmdb',
        r'portal/static/',
        r'portal/reports/',
        '__pycache__',
    )


__all__ = (
    'BackupCerts',
    'BackupDBOSConfig',
    'BackupEtc',
)
