import os
import tarfile
from pathlib import Path

from capsule.backups import ObjectStorageFile, object_server_client
from .settings import S3_REGION, TMP_DIR
from .typehints import INTERVALS, S3_REGIONS
from .utils.config import get_bucket_name


def downloader(
    interval: INTERVALS,
    filename: str,
    base_prefix: str,
    region: S3_REGIONS = S3_REGION,
):
    """
    Download specified backup tarball from the backup server.
    """
    client = object_server_client(region)
    dst = ObjectStorageFile(filename, prefix=f'{base_prefix}/{interval}', client=client).download()

    if not Path(dst).exists():
        print(f'Unable to download {filename}.')
        return

    destination = f'{TMP_DIR}/{os.path.splitext(filename)[0]}'
    Path(destination).mkdir(exist_ok=True)
    with tarfile.open(dst) as tarball:
        tarball.extractall(path=destination)

    os.unlink(dst)

    print(f'Downloaded and extracted {filename} into {destination}')
    print('Browse and select files to manually restore.')


def download_latest(
    base_name: str,
    prefix: str,
    bucket: str = '',
    region: S3_REGIONS = S3_REGION,
) -> str:
    """
    Download the latest file from S3 Object Storage.

    :param base_name: The name of the file to download, without date.
        IE "drawbridge", which will match "drawbridge-20250117.db"
    :param prefix: The parent directory name, such as "databases"
    :param bucket: The bucket name, usually the appliance / server name, such as "thinkwell".
    :param region: The S3 region name, such as "ironwood".
    """
    bucket_name = bucket or get_bucket_name()
    client = object_server_client(region)
    objects = [
        ob
        for ob in client.list_objects(bucket_name, prefix=prefix, recursive=True)
        if base_name in ob.object_name
    ]
    objects.sort(key=lambda o: o.last_modified, reverse=True)

    latest_object = objects[0].object_name.split('/')
    filename = latest_object[-1]
    prefix = '/'.join(latest_object[:-1])

    return ObjectStorageFile(filename, prefix=prefix, client=client).download()


__all__ = (
    'downloader',
    'download_latest',
)
