import arrow
from console_keyring.console_keyring import ConsoleKeyring
from datetime import datetime
import logging
from minio import Minio
from minio.deleteobjects import DeleteObject
from minio.error import S3Error
from pathlib import Path

from capsule.settings import S3_ENDPOINT, S3_REGION, REGION_ENDPOINT_MAP, TMP_DIR
from capsule.typehints import KeyringService
from capsule.utils.config import get_bucket_name

logger = logging.getLogger(__name__)


# --------------------------------------------------------------------------------
class ObjectStorageFile:
    """
    Base class to handle file transfers to and from remote object server.
    """

    def __init__(self, filename: str, prefix: str = '', bucket: str = '', client=None):
        if not (bucket_name := bucket or get_bucket_name()):
            raise SystemExit('No bucket name was defined. Confirm that system setup is complete.')

        self.filename = Path(filename)
        base_name = self.filename.name.split('-')[0]
        self.prefix = f'{prefix}/{base_name}'
        self.object_name = f'{self.prefix}/{self.filename.name}'
        self.bucket = bucket_name
        self._client = client

    def client(self) -> Minio:
        if not self._client:
            self._client = object_server_client(S3_REGION)
        return self._client

    def upload(self) -> str:
        """
        Upload database to remote backup server.
        """
        client = self.client()

        try:
            if not client.bucket_exists(self.bucket):
                client.make_bucket(self.bucket)

            client.fput_object(
                bucket_name=self.bucket,
                object_name=self.object_name,
                file_path=str(self.filename),
            )

        except S3Error as e:
            logger.exception(f'Unable to upload {self.filename}: {e}')

        return 'done'

    def download(self) -> str:
        """
        Download file to destination.
        """
        dst = f'{TMP_DIR}/{self.object_name}'
        self.client().fget_object(
            bucket_name=self.bucket,
            object_name=self.object_name,
            file_path=dst,
        )
        return dst

    def delete(self, objects: list[str]):
        """
        Delete a list objects from the server.
        """
        try:
            resp = self.client().remove_objects(
                bucket_name=self.bucket,
                delete_object_list=[DeleteObject(obj) for obj in objects],
            )
            errors = 0
            for error in resp:
                errors += 1
                logger.info(f'Error deleting object: {error}')
            errors = errors if errors else 'no'
            logger.info(
                f'Deleted {len(objects)} files from {self.bucket}/{self.prefix} with {errors} errors'
            )
        except S3Error as e:
            logger.exception(f'Deleted {len(objects)} from {self.bucket}/{self.prefix}: {e}')

    def scrub(self, max_file_count: int, max_age: arrow.Arrow | datetime):
        """
        Scrub all files older than maximum age if total file count
        exceeds max_file_count and age exceeds max_age.
        """
        obsolete_files = []

        objects = list(
            self.client().list_objects(
                bucket_name=self.bucket,
                prefix=self.prefix,
                recursive=True,
            )
        )

        backup_count = len(objects)
        if backup_count <= max_file_count:
            return

        objects.sort(key=lambda o: o.last_modified)
        oldest_files_over_max_count = objects[backup_count:]
        for obj in oldest_files_over_max_count:
            if obj.last_modified < max_age:
                obsolete_files.append(obj.object_name)

        if obsolete_files:
            self.delete(obsolete_files)

    def list_objects(self, prefix: str):
        """
        Get list of all file objects on the remote server.
        """
        objects = list(self.client().list_objects(self.bucket, prefix=prefix, recursive=True))
        objects.sort(key=lambda o: (o.last_modified, o.object_name), reverse=True)
        return objects


# --------------------------------------------------------------------------------
def object_server_client(region: str) -> Minio:
    """
    Wrapper function to generate Minio client for accessing S3 Object Server,
    to enable custom endpoints and custom regions to be provided.
    """
    kr = ConsoleKeyring()
    if not region:
        endpoint = S3_ENDPOINT
    else:
        try:
            endpoint = REGION_ENDPOINT_MAP[region]
        except KeyError:
            regions = ','.join(REGION_ENDPOINT_MAP)
            raise SystemExit(f'Invalid region. Choose from {regions!r}.')

    return Minio(
        endpoint=endpoint,
        access_key=kr.get_password(KeyringService.LocalConsole, 'access_key'),
        secret_key=kr.get_password(KeyringService.LocalConsole, 'secret_key'),
        region=region,
    )


__all__ = (
    'ObjectStorageFile',
    'object_server_client',
)
