from pathlib import PosixPath


# --------------------------------------------------------------------------------
def strtobool(val: str) -> bool:
    """Convert a string representation of truth to True or False.

    True values are 'y', 'yes', 't', 'true', 'on', and '1'; false values
    are 'n', 'no', 'f', 'false', 'off', and '0'.  Raises ValueError if
    'val' is anything else.
    """
    if not val:
        return False

    val = val.lower()
    if val in ('y', 'yes', 't', 'true', 'on', '1'):
        return True
    elif val in ('n', 'no', 'f', 'false', 'off', '0'):
        return False
    else:
        raise ValueError(f"invalid truth value {val}")


# --------------------------------------------------------------------------------
def get_database_service_name():
    """
    Various versions of Postgres will be installed on, so
    get postgres database name so service can be restarted
    """
    for file_path in ('/usr/lib/systemd/system/', '/etc/init.d/'):
        path = PosixPath(file_path)
        if not path.exists():
            continue

        try:
            script = list(path.glob('postgresql*'))[0]
        except IndexError:
            continue

        # Trim off the .service / .sh extension
        if script.name.endswith(('.sh', 'service')):
            return '.'.join(script.name.split('.')[:-1])
        return script.name

    print('Unable to retrieve database service name')


__all__ = (
    'strtobool',
    'get_database_service_name',
)
