import logging
import os
import pwd
import tarfile

from .base import BackupFile
from ..settings import ETC_DIR, OPT_DIR

logger = logging.getLogger(__name__)


# --------------------------------------------------------------------------------
class BackupDirectory(BackupFile):
    """
    Compress directory to bz2 tarball and backup to remote server.

    Create daily / weekly / monthly backup tarballs, and
    maintain designated archive of copies on the remote server
    """

    directory = ''

    def process(self) -> None:
        """
        Perform all directory tar, upload and cleanup operations.
        """
        self.tar()
        super().process()

    def tar(self) -> None:
        """
        Tar directory to local tarball
        """
        if not os.path.exists(self.file_path):
            os.makedirs(self.file_path, exist_ok=True)

        with tarfile.open(self.filename(), 'w:bz2') as tarball:
            for root, dirs, files in os.walk(self.directory):
                for file in files:
                    tarball.add(os.path.join(root, file))

        # set owner to nginx if that user exists on the system, so the
        # directory can be served via nginx web server for list viewing
        try:
            pwd.getpwnam('nginx')
            self.ctx.run(f'chown -R nginx:nginx {self.file_path}', pty=True)
        except KeyError:
            pass


class BackupCerts(BackupDirectory):
    bucket = 'cert_backups'
    file_path = f'{OPT_DIR}/{bucket}'
    base_prefix = 'certs'
    directory = '/opt/aspen_portal/files'


class BackupDBOSConfig(BackupDirectory):
    bucket = 'dbosbackups'
    file_path = f'{OPT_DIR}/{bucket}'
    base_prefix = 'dbos'
    directory = '/config'


class BackupEtc(BackupDirectory):
    bucket = 'etcbackups'
    file_path = f'{OPT_DIR}/{bucket}'
    base_prefix = 'etc'
    directory = ETC_DIR


__all__ = (
    'BackupCerts',
    'BackupDBOSConfig',
    'BackupEtc',
)
