import logging
from collections import defaultdict
from dataclasses import dataclass
import os
import pwd

from .base import BackupFile
from .object_storage_server import object_server_client
from ..settings import DATABASE_NAME, OPT_DIR, S3_REGION
from ..utils.config import get_bucket_name

logger = logging.getLogger(__name__)


@dataclass(frozen=True)
class DatabaseBackup:
    name: str
    file_path: str


# --------------------------------------------------------------------------------
# Exclude tables from databases when creating backup dump files
DATABASE_EXCLUDE_TABLES = {
    DATABASE_NAME: (
        "django_session",
        "console_sessions_session",
        "overrides_autofixes",
        "overrides_overblocks",
        "reporter_logline*",
        "reporter_summaryantivirus*",
        "reporter_summaryapplication*",
        "reporter_summaryclassifier*",
        "reporter_summarydomains*",
        "reporter_summarydomainsblocked*",
        "reporter_summaryhits*",
        "reporter_summarywebviews*",
        "mailer_message",
        "mailer_messagelog",
        # Exclude Concord Event tables that tend to get large
        "categories_categoryconcordevent",
        "categories_patternconcordevent",
        "mediaroom_channelcategoriesconcordevent",
        "mediaroom_classifiedmediaconcordevent",
    ),
}


# --------------------------------------------------------------------------------
class BackupDatabase(BackupFile):
    """
    Dump Database to file and backup to remote server.

    Create daily / weekly / monthly database files, and
    maintain designated archive of copies on the remote server
    """

    file_path = f'{OPT_DIR}/pgbackups'
    base_prefix = 'databases'

    def filename(self) -> str:
        """
        Standardize the backup file name to 'drawbridge' to
        facilitate upgrading from ClearOS to DrawBridgeOS.
        """
        filename = super().filename()
        if 'logcabin' in filename:
            return filename.replace('logcabin', 'drawbridge')
        return filename

    def process(self) -> None:
        """
        Perform all database dump, upload and cleanup operations.
        """
        self.dump()
        super().process()

    def dump(self) -> None:
        """
        Dump database to local file.
        """
        if not os.path.exists(self.file_path):
            os.makedirs(self.file_path, exist_ok=True)

        database_name = self.base_filename
        base_cmd = f'pg_dump -U postgres -F c {database_name}'

        exclude_tables: tuple[str, ...] = DATABASE_EXCLUDE_TABLES.get(database_name, ())
        if exclude_tables:
            tables = " ".join([f'--exclude-table-data "{table}"' for table in exclude_tables])
            base_cmd = f'{base_cmd} {tables}'

        cmd = f'{base_cmd} -f {self.filename()}'
        self.ctx.run(cmd, pty=True)

        # set owner to nginx if that user exists on the system, so the
        # directory can be served via nginx web server for list viewing
        try:
            pwd.getpwnam('nginx')
            self.ctx.run(f'chown -R nginx:nginx {self.file_path}', pty=True)
        except KeyError:
            pass

    def db_list(self, interval: str) -> list[str]:
        """
        Get list of all database backups on the remote server.
        """
        names = set()
        prefix = self.base_prefix if not interval else f'{self.base_prefix}/{interval}'
        backups = self.server.list_objects(prefix=prefix)

        for backup in backups:
            backup_name = backup.object_name.split('/')[-1].split('-')[0]
            names.add(backup_name)

        return sorted(list(names))


def latest_database_backups(client=None) -> list[DatabaseBackup]:
    """
    Get all database names that have backups on the Garage S3 server.
    """
    client = client or object_server_client(S3_REGION)
    if not (bucket_name := get_bucket_name()):
        return []

    database_objects = client.list_objects(
        bucket_name,
        prefix=BackupDatabase.base_prefix,
        recursive=True,
    )

    database_backups = defaultdict(list)
    for db_file in database_objects:
        # databases/daily/draw-bridge-20250116.db
        file_parts = db_file.object_name.split('/')
        filename = file_parts[-1]
        db_name = '-'.join(filename.split('-')[:-1])
        database_backups[db_name].append(db_file)

    backup_names = []
    for db_name, backups in database_backups.items():
        backups.sort(key=lambda b: b.last_modified, reverse=True)
        latest_backup = backups[0]
        # Send path without the prefix name
        file_path = '/'.join(latest_backup.object_name.split('/')[1:])
        backup_names.append(DatabaseBackup(db_name, file_path))

    return backup_names


__all__ = (
    'BackupDatabase',
    'latest_database_backups',
)
