import base64
from hashlib import sha256
from pyotp import TOTP
from aspen_crypto.encode import mask


def generate_totp(secret: str, digits: int = 6) -> TOTP:
    """
    Generate one-time password, with some obfuscation of the secret
    to make it at least somewhat harder guessing OTPs out of band.
    """
    # remove all spaces and newlines from secret to normalize formatting
    spaceless_secret = "".join(secret.split())
    encoded_name = base64.b32encode(mask(spaceless_secret).encode()).decode()

    # set interval here in this function, so that anywhere it's used
    # the value will be updated if this value is updated. The disadvantage
    # is that it can't be changed per application code...
    interval = 60

    return TOTP(encoded_name, digits=digits, digest=sha256, interval=interval)


def verify_totp(name: str, otp: str, valid_window: int = 5) -> bool:
    """
    Verify that name was correctly encoded with one-time value.
    """
    totp = generate_totp(name, digits=len(otp))
    return totp.verify(otp, valid_window=valid_window)


__all__ = (
    'generate_totp',
    'verify_totp',
)
