from functools import lru_cache
from hashlib import sha3_224
from lcconfig import ConsoleSettingsConfig, SymmetricKeysConfig
from lclazy import LazyLoader
import os
from pathlib import Path
from typing import NamedTuple
from uuid import UUID

from .keys import Ed25519PrivateKey, PrivateKey, FacadePrivateKey, FacadePublicKey
from ..settings import (
    ENCRYPTION_SECTION,
    LOCAL_UMBRELLA_KEY_CID,
    LOCAL_UMBRELLA_KEY_KID,
    LOCAL_VERIFICATION_KEY_FILE,
)

pyseto = LazyLoader('pyseto', globals(), 'pyseto')


class KeyDetails(NamedTuple):
    """
    Canonical ID and Key of Private Key to sign Paseto Tokens.
    """

    cid: str | UUID
    kid: str
    key: FacadePrivateKey
    public_key: FacadePublicKey


def load_private_key(name: str) -> str:
    """
    Load private key prefix and return hashed value.

    Function used for bootstrapping new DrawBridge
    installations, when key is being retrieved,
    saved to disk and then loaded for use elsewhere.

    Use this function in the "elsewhere" call-site to
    read the value from disk.
    """

    sym_keys = SymmetricKeysConfig().as_typed_dict()

    if name not in sym_keys:
        return ''

    return sha3_224(sym_keys.get(name, '').encode()).hexdigest()[12:48]


def get_key_password(cid: str | UUID, password_prefix: str = '', ssh: bool = False) -> bytes:
    """
    Return Private Key password with support for OpenSSH-formatted
    private keys which don't support passwords over 72 bytes.
    """
    if isinstance(cid, str):
        cid = str(cid).replace('-', '')
    else:
        cid = cid.hex

    # When bootstrapping consoles, lookup value in config file, in case
    # it's been generated just now as part of the initial setup process
    if not password_prefix:
        password_prefix = load_private_key('chacha')
        if not password_prefix:
            return b''

    password = f'{password_prefix}|{cid[::-1]}'.encode()

    if ssh and len(password) > 72:
        return password[-72:]

    return password


@lru_cache(maxsize=None)
def load_local_umbrella_api_key() -> KeyDetails:
    """
    The local Encryption Key should be loadable from a file,
    so that scripts can use it without initializing the whole
    Portal to make a database call.
    """
    cfg = ConsoleSettingsConfig().as_typed_dict(ENCRYPTION_SECTION)
    if not (cid := cfg.get(LOCAL_UMBRELLA_KEY_CID)):
        raise ValueError('Local verification key does not exist')

    password = get_key_password(cid, ssh=True)

    try:
        pk = PrivateKey.load_pem_from_file(umbrella_verification_key_file(), password=password)
    except (FileNotFoundError, ValueError, TypeError):
        raise ValueError('Unable to load local verification key') from None

    return KeyDetails(cid, cfg.get(LOCAL_UMBRELLA_KEY_KID), pk, pk.public_key)


def persist_umbrella_api_key(
    key_cid: str | UUID,
    private_key: Ed25519PrivateKey,
) -> KeyDetails:
    """
    Save Umbrella API Private Key to config file.
    """
    password = get_key_password(key_cid, ssh=True)
    with open(umbrella_verification_key_file(), 'wb') as tkf:
        tkf.write(private_key.private_bytes(password))

    public_key = private_key.public_key
    kid = generate_kid(public_key.as_pem)
    cfg = ConsoleSettingsConfig()
    cfg.save_section(
        ENCRYPTION_SECTION,
        data={
            LOCAL_UMBRELLA_KEY_CID: key_cid,
            LOCAL_UMBRELLA_KEY_KID: kid,
        },
    )

    # Ensure cache gets cleared so that regenerating
    # the EncryptionKey will re-read the config file
    load_local_umbrella_api_key.cache_clear()

    return KeyDetails(key_cid, kid, private_key, public_key)


def umbrella_verification_key_file() -> Path:
    """
    Get path to the private key used
    to sign Umbrella API tokens.
    """
    if not (CONF_DIR := os.environ.get('CONF_DIR')):
        try:
            from system_env import setup_portal_environment

            env = setup_portal_environment()
            CONF_DIR = env['CONF_DIR']
        except (ImportError, ModuleNotFoundError, KeyError):
            raise ValueError('Unable to load local verification key') from None

    return Path(f"{CONF_DIR}/{LOCAL_VERIFICATION_KEY_FILE}")


def generate_kid(key: str | bytes) -> str:
    """
    Generate Paseto-based Key ID from the key string.

    v2 & v4 take Ed25519 keys. v1 takes RSA. v3 takes ECDSA.
    In descending order of likely key type.
    """
    PasetoKey = pyseto.Key
    for version in (4, 1, 3):
        try:
            return PasetoKey.new(
                version=version,
                purpose="public",
                key=key,
            ).to_paserk_id()
        except ValueError:
            continue

    return ''


__all__ = (
    'KeyDetails',
    'load_private_key',
    'get_key_password',
    'persist_umbrella_api_key',
    'load_local_umbrella_api_key',
    'umbrella_verification_key_file',
    'generate_kid',
)
