import binascii
from cryptography.fernet import Fernet
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives.kdf.pbkdf2 import PBKDF2HMAC
from functools import lru_cache

try:
    from pybase64 import b64decode, urlsafe_b64encode
except (ImportError, ModuleNotFoundError):
    from base64 import b64decode, urlsafe_b64encode


@lru_cache(32)
def fernet_key(base: str = '', salt: str = '') -> Fernet:
    """
    Generate Fernet key to encrypt / decrypt strings
    """

    if base and salt:
        kdf = PBKDF2HMAC(
            algorithm=hashes.SHA512(),
            length=32,
            salt=bytes(salt, encoding='utf8'),
            iterations=200000,
            backend=default_backend(),
        )
        key = urlsafe_b64encode(kdf.derive(bytes(f'{base.upper()}_{base[::-1]}', encoding='utf8')))
    else:
        from .secrets import KEY

        key = binascii.unhexlify(b64decode(KEY)).swapcase()

    return Fernet(key)


def encrypt(string: bytes | str, key: Fernet | None = None) -> str:
    """
    Symmetrically encrypt strings to save in database.

    :param string: Alphanumeric string to encrypt
    :param key: Fernet Key, pre-generated for efficiency
    """
    if not key:
        key = fernet_key()

    byte_string = string.encode() if isinstance(string, str) else string

    encrypted = key.encrypt(byte_string)

    return encrypted.decode(encoding='utf8')


def decrypt(string: bytes | str, key: Fernet | None = None) -> str:
    """
    Decrypt username/password string from database

    :param string: Alphanumeric string to decrypt
    :param key: Fernet Key, pre-generated for efficiency
    """
    if not key:
        key = fernet_key()

    byte_string = string.encode() if isinstance(string, str) else string

    decrypted = key.decrypt(byte_string)

    return decrypted.decode(encoding='utf8')


__all__ = (
    'decrypt',
    'encrypt',
    'fernet_key',
)
